// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "opcode.h"
#include "registers.h"
#include "stringCommons.h"
#include <stdlib.h>
#include <float.h>

namespace disassembly {
	RegistersBase r;
	DWORD opcode, cia;
}
using namespace disassembly;

void disasm_stw(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_andi(Container<char>& buffer) {
	D_rS; D_rA; D_UIMM;
	sprintf_s(buffer, ". r%i, r%i, 0x%04X", rA, rS, UIMM);
}

void disasm_addi(Container<char>& buffer) {
	D_rD; D_rA; D_SIMM;
	sprintf_s(buffer, " r%i, %s%i, %s0x%04X", rD, rA_0a, rA,
		signh(SIMM) ? "-" : "", abs(SIMM));
}

void disasm_stwu(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (r%i%s0x%04X)", rS, rA, signw(d) ? "-" : "+", abs(d));
	if(rA == 0)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_b(Container<char>& buffer) {
	D_LI; D_AA; D_LK;
	sprintf_s(buffer, "%s%s 0x%08X(%s0x%08X)", LK ? "l" : "", AA ? "a" : "",
		AA ? LI : cia + LI, AA ? "" : (signw(LI) ? "-" : "+"), AA ? LI : abs(LI));
}

void disasm_addis(Container<char>& buffer) {
	D_rD; D_rA; D_SIMM;
	sprintf_s(buffer, " r%i, %s%i, 0x%04X", rD, rA_0a, rA, (WORD)SIMM);
}

void disasm_ori(Container<char>& buffer) {
	D_rS; D_rA; D_UIMM;
	sprintf_s(buffer, " r%i, r%i, 0x%04X", rA, rS, UIMM);
}

void disasm_bclr(Container<char>& buffer) {
	D_BO; D_BI; D_LK;
	sprintf_s(buffer, "%s %s(0x%02X), %s(%i)", LK ? "l" : "", g_asms.getBOasm(BO).c_str(), BO,
		g_asms.getBIasm(BI).c_str(), BI);
}

void disasm_dcbf(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " %s%i, r%i\t//NOP", rA_0a, rA, rB);
}

void disasm_mfmsr(Container<char>& buffer) {
	D_rD;
	sprintf_s(buffer, " r%i", rD);
}

void disasm_mtmsr(Container<char>& buffer) {
	D_rS;
	sprintf_s(buffer, " r%i", rS);
}

void disasm_mfspr(Container<char>& buffer) {
	D_rD; D_spr;
	sprintf_s(buffer, " r%i, %s (%i)", rD, r.getsprasm(spr).c_str(), spr);
	if(!r.mfspr_is_valid(spr))
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_oris(Container<char>& buffer) {
	D_rS; D_rA; D_UIMM;
	sprintf_s(buffer, " r%i, r%i, 0x%04X", rA, rS, UIMM);
}

void disasm_or(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_mtspr(Container<char>& buffer) {
	D_rS; D_spr;
	sprintf_s(buffer, " %s(%i), r%i", r.getsprasm(spr).c_str(), spr, rS);
	if(!r.mtspr_is_valid(spr)) {
		DEGUB("%s\n", buffer);
		throw bad_form_exception(bad_form_exception::Invalid);
	}
}

void disasm_bc(Container<char>& buffer) {
	D_BO; D_BI; D_BD; D_AA; D_LK;
	int i = sprintf_s(buffer, "%s%s %s(0x%02X), %s(%i), 0x%08X", LK ? "l" : "",
		AA ? "a" : "", g_asms.getBOasm(BO).c_str(), BO, g_asms.getBIasm(BI).c_str(), BI,
		AA ? BD : cia + BD);
	if(!AA) {
		Container<char>::SubContainer sub = buffer.getSub(i);
		sprintf_s(sub, "(%s0x%04X)", signw(BD) ? "-" : "+", abs(BD));
	}
}

void disasm_lwz(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_sync(Container<char>& buffer) {
	sprintf_s(buffer, "\t//NOP");
}

void disasm_rlwinm(Container<char>& buffer) {
	D_rS; D_rA; D_SH; D_MB; D_ME; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, %i, %i, %i", Rca, rA, rS, SH, MB, ME);
}

void disasm_cmpli(Container<char>& buffer) {
	D_crfD; D_L; D_rA; D_UIMM;
	sprintf_s(buffer, " cr%i, %i, r%i, 0x%04X", crfD, L, rA, UIMM);
	if(L)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_isync(Container<char>& buffer) {
	sprintf_s(buffer, "\t//NOP");
}

void disasm_crxor(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_add(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_subf(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_cmp(Container<char>& buffer) {
	D_crfD; D_L; D_rA; D_rB;
	sprintf_s(buffer, " cr%i, %i, r%i, r%i", crfD, L, rA, rB);
	if(L)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_xori(Container<char>& buffer) {
	D_rS; D_rA; D_UIMM;
	sprintf_s(buffer, " r%i, r%i, 0x%04X", rA, rS, UIMM);
}

void disasm_cmpi(Container<char>& buffer) {
	D_crfD; D_L; D_rA; D_SIMM;
	sprintf_s(buffer, " cr%i, %i, r%i, %s0x%04X", crfD, L, rA, signh(SIMM) ? "-" : "",
		abs(SIMM));
	if(L)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_lhz(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_stb(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_lbz(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_mulli(Container<char>& buffer) {
	D_rD; D_rA; D_SIMM;
	sprintf_s(buffer, " r%i, r%i, %s0x%04X", rD, rA,
		signh(SIMM) ? "-" : "", abs(SIMM));
}

void disasm_mulhw(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rD, rA, rB);
}

void disasm_srawi(Container<char>& buffer) {
	D_rS; D_rA; D_SH; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, %i", Rca, rA, rS, SH);
}

void disasm_stfd(Container<char>& buffer) {
	D_frS; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_lfd(Container<char>& buffer) {
	D_frD; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_extsb(Container<char>& buffer) {
	D_rS; D_rA; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i", Rca, rA, rS);
}

void disasm_mtfsb1(Container<char>& buffer) {
	D_crbD; D_Rc;
	sprintf_s(buffer, "%s %s(%i)", Rca, r.getfpscrasm(crbD), crbD);
}

void disasm_andc(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_cntlzw(Container<char>& buffer) {
	D_rS; D_rA; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i", Rca, rA, rS);
}

void disasm_sth(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_and(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_xor(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_neg(Container<char>& buffer) {
	D_rD; D_rA; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i", OE ? "o" : "", Rca, rD, rA);
}

void disasm_extsh(Container<char>& buffer) {
	D_rS; D_rA; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i", Rca, rA, rS);
}

void disasm_addze(Container<char>& buffer) {
	D_rD; D_rA; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i", OE ? "o" : "", Rca, rD, rA);
}

void disasm_fmr(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_mtfsf(Container<char>& buffer) {
	D_FM; D_frB; D_Rc;
	sprintf_s(buffer, "%s 0x%02X, fr%i", Rca, FM, frB);
}

void disasm_lwzx(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_subfic(Container<char>& buffer) {
	D_rD; D_rA; D_SIMM;
	sprintf_s(buffer, " r%i, r%i, %s0x%04X", rD, rA, signh(SIMM) ? "-" : "", abs(SIMM));
}

void disasm_lfs(Container<char>& buffer) {
	D_frD; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_stwx(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rS, rA_0a, rA, rB);
}

void disasm_xoris(Container<char>& buffer) {
	D_rS; D_rA; D_UIMM;
	sprintf_s(buffer, " r%i, r%i, 0x%04X", rA, rS, UIMM);
}

void disasm_fsub(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_frsp(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_addic(Container<char>& buffer) {
	D_rD; D_rA; D_SIMM;
	sprintf_s(buffer, " r%i, r%i, %s0x%04X", rD, rA, signh(SIMM) ? "-" : "", abs(SIMM));
}

void disasm_subfe(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_lbzx(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_mullw(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_addic_(Container<char>& buffer) {
	D_rD; D_rA; D_SIMM;
	sprintf_s(buffer, " r%i, r%i, %s0x%04X", rD, rA, signh(SIMM) ? "-" : "", abs(SIMM));
}

void disasm_lha(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_stbx(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rS, rA_0a, rA, rB);
}

void disasm_sthx(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rS, rA_0a, rA, rB);
}

void disasm_bcctr(Container<char>& buffer) {
	D_BO; D_BI; D_LK;
	sprintf_s(buffer, "%s %s(0x%02X), %s(%i)", LK ? "l" : "",
		g_asms.getBOasm(BO).c_str(), BO, g_asms.getBIasm(BI).c_str(), BI);
	if(!getbitr(BO, 2))
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_slw(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_sraw(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_divw(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_adde(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_cmpl(Container<char>& buffer) {
	D_crfD; D_L; D_rA; D_rB;
	sprintf_s(buffer, " cr%i, %i, r%i, r%i", crfD, L, rA, rB);
	if(L)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_mcrf(Container<char>& buffer) {
	D_crfD; D_crfS;
	sprintf_s(buffer, " cr%i, cr%i", crfD, crfS);
}

void disasm_mfcr(Container<char>& buffer) {
	D_rD;
	sprintf_s(buffer, " r%i", rD);
}

void disasm_mtcrf(Container<char>& buffer) {
	D_rS; D_CRM;
	sprintf_s(buffer, " 0x%02X, r%i", CRM, rS);
}

void disasm_fctiwz(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_stfs(Container<char>& buffer) {
	D_frS; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_fdivs(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_fadd(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_fcmpu(Container<char>& buffer) {
	D_crfD; D_frA; D_frB;
	sprintf_s(buffer, " cr%i, fr%i, fr%i", crfD, frA, frB);
}

void disasm_fmul(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frC);
}

void disasm_fmadd(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_fadds(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_fsubs(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_fmadds(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_fmuls(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frC);
}

void disasm_cror(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_subfc(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_lbzu(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (r%i%s0x%04X)", rD, rA, signw(d) ? "-" : "+", abs(d));
	if(rA == 0 || rA == rD)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_nor(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_lwzu(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (r%i%s0x%04X)", rD, rA, signw(d) ? "-" : "+", abs(d));
	if(rA == 0 || rA == rD)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_dcbst(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " %s%i, r%i\t//NOP", rA_0a, rA, rB);
}

void disasm_sc(Container<char>& buffer) {
	sprintf_s(buffer, "");
}

void disasm_mftb(Container<char>& buffer) {
	D_rD; D_spr;
	sprintf_s(buffer, " r%i, %s (%i)", rD, r.getsprasm(spr).c_str(), spr);
	if(!r.mfspr_is_valid(spr) || !(spr == 268 || spr == 269))
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_addc(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_stmw(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_icbi(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " %s%i, r%i", rA_0a, rA, rB);
}

void disasm_lmw(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
	if(rA >= rD)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_lhzx(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_mulhwu(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rD, rA, rB);
}

void disasm_rlwimi(Container<char>& buffer) {
	D_rS; D_rA; D_SH; D_MB; D_ME; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, %i, %i, %i", Rca, rA, rS, SH, MB, ME);
}

void disasm_dcbi(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " %s%i, r%i\t//NOP", rA_0a, rA, rB);
}

void disasm_stbu(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (r%i%s0x%04X)", rS, rA, signw(d) ? "-" : "+", abs(d));
	if(rA == 0)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_nand(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_rfi(Container<char>& buffer) {
	sprintf_s(buffer, "");
}

void disasm_divwu(Container<char>& buffer) {
	D_rD; D_rA; D_rB; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i, r%i", OE ? "o" : "", Rca, rD, rA, rB);
}

void disasm_srw(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_fmsubs(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_stfsx(Container<char>& buffer) {
	D_frS; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frS, rA_0a, rA, rB);
}

void disasm_lhax(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_lfsx(Container<char>& buffer) {
	D_frD; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frD, rA_0a, rA, rB);
}

void disasm_fneg(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_lfdx(Container<char>& buffer) {
	D_frD; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frD, rA_0a, rA, rB);
}

void disasm_addme(Container<char>& buffer) {
	D_rD; D_rA; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i", OE ? "o" : "", Rca, rD, rA);
}

void disasm_crnor(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_fdiv(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_andis(Container<char>& buffer) {
	D_rS; D_rA; D_UIMM;
	sprintf_s(buffer, ". r%i, r%i, 0x%04X", rA, rS, UIMM);
}

void disasm__undefined(Container<char>& buffer) {
	sprintf_s(buffer, " 0x%08X", opcode);
}

void disasm_twi(Container<char>& buffer) {
	D_TO; D_rA; D_SIMM;
	sprintf_s(buffer, " 0x%02X, r%i, %s0x%04X", TO, rA,
		signh(SIMM) ? "-" : "", abs(SIMM));
}

void disasm_fmsub(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_subfze(Container<char>& buffer) {
	D_rD; D_rA; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i", OE ? "o" : "", Rca, rD, rA);
}

void disasm_subfme(Container<char>& buffer) {
	D_rD; D_rA; D_OE; D_Rc;
	sprintf_s(buffer, "%s%s r%i, r%i", OE ? "o" : "", Rca, rD, rA);
}

void disasm_orc(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_lhzu(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_lfdu(Container<char>& buffer) {
	D_frD; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_dcbz(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " %s%i, r%i\t//NOP", rA_0a, rA, rB);
}

void disasm_lbzux(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
	if(rA == 0 || rA == rD)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_psq_lu(Container<char>& buffer) {
	D_frD; D_rA; D_W; D_I; D_d20;
	sprintf_s(buffer, " fr%i, (r%i%s0x%04X), %i, %i", frD,
		rA, signw(d) ? "-" : "+", abs(d), W, I);
	if(rA == 0)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_sthu(Container<char>& buffer) {
	D_rS; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (r%i%s0x%04X)", rS, rA, signw(d) ? "-" : "+", abs(d));
	if(rA == 0)
		throw bad_form_exception(bad_form_exception::Invalid);
}

void disasm_fnmadd(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_creqv(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_fabs(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_mtsrin(Container<char>& buffer) {
	D_rS; D_rB;
	sprintf_s(buffer, " r%i, r%i", rS, rB);
}

void disasm_mfsr(Container<char>& buffer) {
	D_rD; D_SR;
	sprintf_s(buffer, " r%i, SR%i", rD, SR);
}

void disasm_mffs(Container<char>& buffer) {
	D_frD; D_Rc;
	sprintf_s(buffer, "%s fr%i", Rca, frD);
}

void disasm_tlbie(Container<char>& buffer) {
	D_rB;
	sprintf_s(buffer, " r%i", rB);
}

void disasm_stswi(Container<char>& buffer) {
	D_rS; D_rA; D_NB;
	sprintf_s(buffer, " r%i, r%i, %i", rS, rA, NB);
}

void disasm_lwarx(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_stwcx_(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rS, rA_0a, rA, rB);
}

void disasm_lwzux(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, r%i, r%i", rD, rA, rB);
}

void disasm_dcbt(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " r%i, r%i", rA, rB);
}

void disasm_lswi(Container<char>& buffer) {
	D_rD; D_rA; D_NB;
	sprintf_s(buffer, " r%i, r%i, %i", rD, rA, NB);
}

void disasm_dcbtst(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " r%i, r%i", rA, rB);
}

void disasm_rlwnm(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_MB; D_ME; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i, %i, %i", Rca, rA, rS, rB, MB, ME);
}

void disasm_stwux(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, r%i, r%i", rS, rA, rB);
}

void disasm_stwbrx(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rS, rA_0a, rA, rB);
}

void disasm_mfsrin(Container<char>& buffer) {
	D_rD; D_rB;
	sprintf_s(buffer, " r%i, r%i", rD, rB);
}

void disasm_fsel(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_frC; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frB, frC);
}

void disasm_psq_l(Container<char>& buffer) {
	D_frD; D_rA; D_W; D_I; D_d20;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X), %i, %i", frD,
		rA_0a, rA, signw(d) ? "-" : "+", abs(d), W, I);
}

void disasm_psq_st(Container<char>& buffer) {
	D_frS; D_rA; D_W; D_I; D_d20;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X), %i, %i", frS,
		rA_0a, rA, signw(d) ? "-" : "+", abs(d), W, I);
}

void disasm_fcmpo(Container<char>& buffer) {
	D_crfD; D_frA; D_frB;
	sprintf_s(buffer, " cr%i, fr%i, fr%i", crfD, frA, frB);
}

void disasm_ps_merge01(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_ps_merge10(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_ps_mr(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_stfsu(Container<char>& buffer) {
	D_frS; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_stfdx(Container<char>& buffer) {
	D_frS; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frS, rA_0a, rA, rB);
}

void disasm_stbux(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, r%i, r%i", rS, rA, rB);
}

void disasm_ps_mul(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frC);
}

void disasm_ps_madd(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_sum0(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_frsqrte(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_fnmsubs(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm__wc_debug(Container<char>& buffer) {
	sprintf_s(buffer, "");
}

void disasm_ps_muls0(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frC);
}

void disasm_ps_muls1(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frC);
}

void disasm_ps_msub(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_neg(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_lfsu(Container<char>& buffer) {
	D_frD; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_eieio(Container<char>& buffer) {
	sprintf_s(buffer, "\t//NOP");
}

void disasm_ps_merge00(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_ps_merge11(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_fnmadds(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_madds0(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_madds1(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_add(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_ps_sub(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_ps_cmpo0(Container<char>& buffer) {
	D_crfD; D_frA; D_frB;
	sprintf_s(buffer, " cr%i, fr%i, fr%i", crfD, frA, frB);
}

void disasm_ps_cmpo1(Container<char>& buffer) {
	D_crfD; D_frA; D_frB;
	sprintf_s(buffer, " cr%i, fr%i, fr%i", crfD, frA, frB);
}

void disasm_ps_cmpu0(Container<char>& buffer) {
	D_crfD; D_frA; D_frB;
	sprintf_s(buffer, " cr%i, fr%i, fr%i", crfD, frA, frB);
}

void disasm_ps_cmpu1(Container<char>& buffer) {
	D_crfD; D_frA; D_frB;
	sprintf_s(buffer, " cr%i, fr%i, fr%i", crfD, frA, frB);
}

void disasm_fres(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_ps_nmadd(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_nmsub(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_ps_sum1(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_eqv(Container<char>& buffer) {
	D_rS; D_rA; D_rB; D_Rc;
	sprintf_s(buffer, "%s r%i, r%i, r%i", Rca, rA, rS, rB);
}

void disasm_lhau(Container<char>& buffer) {
	D_rD; D_rA; D_d16;
	sprintf_s(buffer, " r%i, (%s%i%s0x%04X)", rD, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_lhaux(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_fnmsub(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_psq_stu(Container<char>& buffer) {
	D_frS; D_rA; D_W; D_I; D_d20;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X), %i, %i", frS,
		rA_0a, rA, signw(d) ? "-" : "+", abs(d), W, I);
}

void disasm_ps_rsqrte(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_ps_res(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm__oovpa(Container<char>& buffer) {
	DWORD i = getbitsw(opcode, 6, 31);
	sprintf_s(buffer, " %i", i);
}

void disasm__gcm_dol_loaded(Container<char>& buffer) {
	sprintf_s(buffer, "");
}

void disasm_mcrxr(Container<char>& buffer) {
	D_crfD;
	sprintf_s(buffer, " cr%i", crfD);
}

void disasm_stfiwx(Container<char>& buffer) {
	D_frS; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frS, rA_0a, rA, rB);
}

void disasm_fctiw(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_sthbrx(Container<char>& buffer) {
	D_rS; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rS, rA_0a, rA, rB);
}

void disasm__osreport(Container<char>& buffer) {
	sprintf_s(buffer, "");
}

void disasm_stfdu(Container<char>& buffer) {
	D_frS; D_rA; D_d16;
	sprintf_s(buffer, " fr%i, (%s%i%s0x%04X)", frS, rA_0a, rA, signw(d) ? "-" : "+", abs(d));
}

void disasm_stfdux(Container<char>& buffer) {
	D_frS; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frS, rA_0a, rA, rB);
}

void disasm_mtfsb0(Container<char>& buffer) {
	D_crbD; D_Rc;
	sprintf_s(buffer, "%s %s(%i)", Rca, r.getfpscrasm(crbD), crbD);
}

void disasm_dcbz_l(Container<char>& buffer) {
	D_rA; D_rB;
	sprintf_s(buffer, " %s%i, r%i\t//NOP", rA_0a, rA, rB);
}

void disasm_lwbrx(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_lhbrx(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_lfdux(Container<char>& buffer) {
	D_frD; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frD, rA_0a, rA, rB);
}

void disasm_lfsux(Container<char>& buffer) {
	D_frD; D_rA; D_rB;
	sprintf_s(buffer, " fr%i, %s%i, r%i", frD, rA_0a, rA, rB);
}

void disasm_lhzux(Container<char>& buffer) {
	D_rD; D_rA; D_rB;
	sprintf_s(buffer, " r%i, %s%i, r%i", rD, rA_0a, rA, rB);
}

void disasm_psq_lx(Container<char>& buffer) {
	D_frD; D_rA; D_rB; D_W; D_I;
	sprintf_s(buffer, " fr%i, %s%i, r%i, %i, %i", frD, rA_0a, rA, rB, W, I);
}

void disasm_psq_stx(Container<char>& buffer) {
	D_frS; D_rA; D_rB; D_W; D_I;
	sprintf_s(buffer, " fr%i, %s%i, r%i, %i, %i", frS, rA_0a, rA, rB, W, I);
}

void disasm_psq_lux(Container<char>& buffer) {
	D_frD; D_rA; D_rB; D_W; D_I;
	sprintf_s(buffer, " fr%i, %s%i, r%i, %i, %i", frD, rA_0a, rA, rB, W, I);
}

void disasm_psq_stux(Container<char>& buffer) {
	D_frS; D_rA; D_rB; D_W; D_I;
	sprintf_s(buffer, " fr%i, %s%i, r%i, %i, %i", frS, rA_0a, rA, rB, W, I);
}

void disasm_mtsr(Container<char>& buffer) {
	D_SR; D_rS;
	sprintf_s(buffer, " SR%i, r%i", SR, rS);
}

void disasm_ps_abs(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_ps_div(Container<char>& buffer) {
	D_frD; D_frA; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i", Rca, frD, frA, frB);
}

void disasm_ps_nabs(Container<char>& buffer) {
	D_frD; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i", Rca, frD, frB);
}

void disasm_ps_sel(Container<char>& buffer) {
	D_frD; D_frA; D_frC; D_frB; D_Rc;
	sprintf_s(buffer, "%s fr%i, fr%i, fr%i, fr%i", Rca, frD, frA, frC, frB);
}

void disasm_tlbsync(Container<char>& buffer) {
	sprintf_s(buffer, "\t//NOP");
}

void disasm_mcrfs(Container<char>& buffer) {
	D_crfD; D_crfS;
	sprintf_s(buffer, " cr%i, cr%i", crfD, crfS);
}

void disasm_crand(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_crandc(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_crnand(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm_crorc(Container<char>& buffer) {
	D_crbD; D_crbA; D_crbB;
	sprintf_s(buffer, " %s(%i), %s(%i), %s(%i)", g_asms.getcrbasm(crbD).c_str(), crbD, 
		g_asms.getcrbasm(crbA).c_str(), crbA, g_asms.getcrbasm(crbB).c_str(), crbB);
}

void disasm__dsp_quick_execute(Container<char>& buffer) {
	sprintf_s(buffer, "");
}
